#include "MessagingSony.h"
#include "MessagePipe.h"
#include "ErrorCodesSony.h"
#include "SignInSony.h"

using namespace sce::Toolkit::NP;
using namespace sce::Toolkit::NP::Utilities;

namespace UnityPlugin
{

	extern std::string gSessionImageFilePath;

	CachedMessaging gMessaging;

	char CachedMessaging::sUnityMessageMagic[8] = {'U', 'n', 'i', 't', 'y', 'M', 's', 'g'};

	enum UnityPluginMessageType
	{
		MESSAGE_TYPE_GAME_INVITE,
		MESSAGE_TYPE_GAME_DATA,
	};

	struct UnityPluginMessageHeader
	{
		char magic[8];
		UnityPluginMessageType messageType;
		int dataSize;
	};

	DO_EXPORT( bool, PrxMessagingIsBusy ) ()
	{
		return gMessaging.IsBusy();
	}

	DO_EXPORT( bool, PrxMessagingSendMessage ) (const MsgRequest* request)	//const char* body, int expireMinutes, void* data, int dataSize)
	{
		return gMessaging.SendMessage(request);	//body, expireMinutes, data, dataSize);
	}

	DO_EXPORT( bool, PrxMessagingSendGameInvite ) (const MsgRequest* request)	//const char* body, int expireMinutes, void* data, int dataSize)
	{
		return gMessaging.SendGameInvite(request);	//body, expireMinutes, data, dataSize);
	}
	DO_EXPORT( bool, PrxMessagingSendInGameDataMessage ) (const unsigned char* npID, void* data, int dataSize)
	{
		printf("STUB : PrxMessagingSendInGameDataMessage\n" );
		return false;
	}
	DO_EXPORT( bool, PrxMessagingShowMessageDialog ) ()
	{
		printf("STUB : PrxMessagingShowMessageDialog\n" );
		return false;
	}

	DO_EXPORT( bool, PrxMessagingShowInviteDialog ) ()
	{
		printf("STUB : PrxMessagingShowInviteDialog\n" );
		return false;
	}
	DO_EXPORT( bool, PrxMessagingGetMessageAttachment ) (MessageAttachment* attachment)
	{
		return gMessaging.GetMessageAttachement(attachment);
	}

	DO_EXPORT( bool, PrxMessagingGetGameInviteAttachment) (MessageAttachment* attachment)
	{
		return gMessaging.GetGameInviteAttachement(attachment);
	}

	DO_EXPORT( bool, PrxHasInGameDataMessage ) ()
	{
		return gMessaging.HasInGameDataMessages();
	}

	DO_EXPORT( bool, PrxGetFirstInGameDataMessage ) (InGameDataMessage* message)
	{
		return gMessaging.GetFirstInGameDataMessage(message);
	}

	DO_EXPORT( bool, PrxRemoveFirstInGameDataMessage ) ()
	{
		return gMessaging.RemoveFirstInGameDataMessage();
	}

	CachedMessaging::CachedMessaging()
		: m_Busy(false)
		, m_SessionInviteMessageAttachment(NULL)
		, m_GameInviteAttachment(NULL)
		, m_MessageAttachment(NULL)
	{
		InitMessage();
	}

	CachedMessaging::~CachedMessaging()
	{
		SimpleLock::AutoLock lock(m_Lock);
		free(m_Message.attachment);
		delete m_SessionInviteMessageAttachment;
		delete m_GameInviteAttachment;
		delete m_MessageAttachment;
	}

	void	CachedMessaging::InitMessage( void )
	{
		//	m_Message is of type sce::Toolkit::NP::MessageData
		//	On PS3 this contains non-POD types (STL list)
		//	We must therefore initialize carefully and cannot use memset on a struct with non-POD data elements
		m_Message.attachment = NULL;
		m_Message.attachmentSize = 0;
		m_Message.recipients.clear();
		m_Message.subject = "";
		m_Message.body = "";

	}

	void InGameMessageFIFO::Add(InGameDataMessage& gameDataMsg)
	{
		SimpleLock::AutoLock lock(m_Lock);
		m_FIFO.push_back(gameDataMsg);
	}

	bool InGameMessageFIFO::HasData()
	{
		SimpleLock::AutoLock lock(m_Lock);
		return !m_FIFO.empty();
	}

	bool InGameMessageFIFO::GetFirst(InGameDataMessage* gameDataMsg)
	{
		SimpleLock::AutoLock lock(m_Lock);
		if(!m_FIFO.empty())
		{
			*gameDataMsg = m_FIFO.front();
			return true;
		}

		return false;
	}

	bool InGameMessageFIFO::RemoveFirst()
	{
		SimpleLock::AutoLock lock(m_Lock);
		if(!m_FIFO.empty())
		{
			InGameDataMessage img = m_FIFO.front();
			free(img.data);
			m_FIFO.pop_front();
			return true;
		}

		return false;
	}

	void InGameMessagePendingFIFO::Add(int gameDataMsgID)
	{
		SimpleLock::AutoLock lock(m_Lock);
		m_FIFO.push_back(gameDataMsgID);
	}

	bool InGameMessagePendingFIFO::HasData()
	{
		SimpleLock::AutoLock lock(m_Lock);
		return !m_FIFO.empty();
	}

	bool InGameMessagePendingFIFO::GetFirst(int* gameDataMsgID)
	{
		SimpleLock::AutoLock lock(m_Lock);
		if(!m_FIFO.empty())
		{
			*gameDataMsgID = m_FIFO.front();
			return true;
		}

		return false;
	}

	bool InGameMessagePendingFIFO::RemoveFirst()
	{
		SimpleLock::AutoLock lock(m_Lock);
		if(!m_FIFO.empty())
		{
			m_FIFO.pop_front();
			return true;
		}

		return false;
	}
	#define GETCOMMID attachment->getCommId().data
	bool CachedMessaging::ProcessEvent(const sce::Toolkit::NP::Event& event)
	{
		SimpleLock::AutoLock lock(m_Lock);
		bool handled = false;
		//int ret;

		switch(event.event)
		{
			case Event::messageSent:						// An event generated when a message has been sent.
				switch(event.returnCode)
				{
					case SCE_TOOLKIT_NP_SUCCESS:
						UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_MessagingSent);
						break;

					case 1:
						UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_MessagingCanceled);
						break;

					default:
						UnityPlugin::Messages::LogWarning("messageSent unknown return code - %s\n", LookupSceErrorCode(event.returnCode));
						UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_MessagingNotSent);
						break;
				}

				free(m_Message.attachment);
				m_Message.attachment = NULL;
				m_Message.attachmentSize = 0;

				m_Busy = false;
				handled = true;
				break;
			default:
				UnityPlugin::Messages::LogWarning("Unexpected event from messaging service: event=%d\n", event.event);
				handled = true;
				break;
		}

		return handled;
	}

	bool CachedMessaging::IsBusy()
	{
		SimpleLock::AutoLock lock(m_Lock);
		return m_Busy;
	}

	bool CachedMessaging::SendMessage(const MsgRequest* request)
	{
		void* data = request->data;
		int dataSize = request->dataSize;

		if(data == NULL)
		{
			static char noData[] = "No data";
			data = noData;
			dataSize = strlen(noData)+1;
		}

		InitMessage();
		m_Message.body = request->body;
		int ret;

#if __ORBIS__ || PSP2_USING_WEBAPI
		m_Message.dialogFlag = SCE_TOOLKIT_NP_DIALOG_TYPE_USER_EDITABLE;
		m_Message.dataDescription = request->dataDescription;
		m_Message.dataName = request->dataName;
		m_Message.npIds = NULL;		// Array of npIDs to send the message to.
		m_Message.npIdsCount = request->npIDCount;	// Number of IDs that can be selected in the message dialog, or if npIds != NULL the number of IDs in the array.
		m_Message.iconPath = request->iconPath;
		memset(&m_Message.npSessionId, 0, sizeof(m_Message.npSessionId));
#endif
		
		int msgSize = sizeof(UnityPluginMessageHeader) + dataSize;
		SceChar8* buffer = (SceChar8*)malloc(msgSize);
		UnityPluginMessageHeader* msgHeader = (UnityPluginMessageHeader*)buffer;
		SceChar8* msgData = (SceChar8*)(msgHeader + 1);

		memcpy(msgHeader->magic, sUnityMessageMagic, sizeof(msgHeader->magic));
		msgHeader->messageType = MESSAGE_TYPE_GAME_DATA;
		msgHeader->dataSize = dataSize;
		memcpy(msgData, data, dataSize);
		
		m_Message.attachment = buffer;
		m_Message.attachmentSize = msgSize;

		ret = Messaging::Interface::sendMessage(&m_Message,SCE_TOOLKIT_NP_MESSAGE_TYPE_CUSTOM_DATA);
		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			Messages::LogError("Messaging::%s@L%d - %s", __FUNCTION__, __LINE__, LookupSceErrorCode(ret));
			return false;
		}

		return true;
	}

	// Game invite with custom data. NOTE for session invites use CachedMatching::InviteToSession instead.
	bool CachedMessaging::SendGameInvite(const MsgRequest* request)
	{
		void* data = request->data;
		int dataSize = request->dataSize;
		int expireMinutes = request->expireMinutes;

		if(data == NULL)
		{
			static char noData[] = "No data";
			data = noData;
			dataSize = strlen(noData)+1;
		}

		if(request->expireMinutes <= 0)		// Invites MUST have an expiry time > 0.
		{
			Messages::LogError("Messaging::%s@L%d - %s", __FUNCTION__, __LINE__, "expireMinutes invalid");
			expireMinutes = 30;
		}

		InitMessage();
		m_Message.body = request->body;
		int msgSize = sizeof(UnityPluginMessageHeader) + dataSize;
		SceChar8* buffer = (SceChar8*)malloc(msgSize);
		UnityPluginMessageHeader* msgHeader = (UnityPluginMessageHeader*)buffer;
		SceChar8* msgData = (SceChar8*)(msgHeader + 1);

		memcpy(msgHeader->magic, sUnityMessageMagic, sizeof(msgHeader->magic));
		msgHeader->messageType = MESSAGE_TYPE_GAME_INVITE;
		msgHeader->dataSize = dataSize;
		memcpy(msgData, data, dataSize);

		m_Message.attachment = buffer;
		m_Message.attachmentSize = msgSize;
		
		int ret = Messaging::Interface::sendMessage(&m_Message, SCE_TOOLKIT_NP_MESSAGE_TYPE_INVITE, true);
		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			Messages::LogError("Messaging::%s@L%d - %s", __FUNCTION__, __LINE__, LookupSceErrorCode(ret));
			return false;
		}

		return true;
	}


	sce::Toolkit::NP::MessageAttachment* CachedMessaging::GetSessionInviteAttachment()
	{
		SimpleLock::AutoLock lock(m_Lock);
		return m_SessionInviteMessageAttachment;
	}

	bool CachedMessaging::GetMessageAttachement(MessageAttachment* attachement)
	{
		SimpleLock::AutoLock lock(m_Lock);
		if(m_MessageAttachment)
		{
			attachement->data = m_MessageAttachment->getAttachmentData();
			attachement->dataSize = m_MessageAttachment->getAttachmentSize();
			return true;
		}

		return false;
	}

	bool CachedMessaging::GetGameInviteAttachement(MessageAttachment* attachement)
	{
		SimpleLock::AutoLock lock(m_Lock);
		if(m_GameInviteAttachment)
		{
			attachement->data = m_GameInviteAttachment->getAttachmentData();
			attachement->dataSize = m_GameInviteAttachment->getAttachmentSize();
			return true;
		}

		return false;
	}

	bool CachedMessaging::HasInGameDataMessages()
	{
		return m_InGameDataMessageFIFO.HasData();
	}

	bool CachedMessaging::GetFirstInGameDataMessage(InGameDataMessage* message)
	{
		return m_InGameDataMessageFIFO.GetFirst(message);
	}

	bool CachedMessaging::RemoveFirstInGameDataMessage()
	{
		return m_InGameDataMessageFIFO.RemoveFirst();
	}

	void CachedMessaging::Update()
	{
	}

	// Retreive the message attachment from the invite message
	// ... will trigger a Event::messageRetrieved if successful
	// retrieveMessageAttachmentFromEvent() only exists on PS4


}
